;FIFO Routines

;FIFO_Init			Initialises a FIFO
;FIFO_Read			Reads from a FIFO, returns 0x00 from empty FIFO
;FIFO_Write			Writes to a FIFO, does not write to full FIFO
;FIFO_Write_Circ	Writes to a FIFO, overwrites a full FIFO

;The routines in this file require the FIFO ram be declared as follows.

;FIFO_Name:		.byte (FIFO_Size+6)

;The FIFO Structure is as follows.

;FIFO_Name		Byte Counter
;				Out Address Low
;				Out Address High
;				In Address Low
;				In Address High
;				FIFO Size
;				Byte 0
;				...
;				Byte n

;The FIFO must be initialised as follows.

;Set_FIFO_Address FIFO_Name
;ldi r16,FIFO_Size
;call FIFO_Init

;To read or write to the FIFO, the FIFO_Name location must be written to
;the Z registers. This may be done with a macro as follows.

;Set_FIFO_Address FIFO_Name

;FIFO Reads and Writes do not modify the Z register, allowing multiple
;reads and/or writes without continual Address setting. To read a byte
;from the FIFO, call the Read_FIFO subroutine after setting a valid FIFO_Name
;location. For example...

;Set_FIFO_Address FIFO_Name
;call FIFO_Read

;Data from the FIFO is returned to r16. r16 is set to 0x00 if the FIFO
;was empty before the read operation. To write to the FIFO, call the
;FIFO_Write after setting a valid FIFO_Name location and setting the
;byte to write in the r16 register. For example...

;Set_FIFO_Address FIFO_Name
;ldi r16,0x01
;call FIFO_Write

;If the FIFO is full, the byte in r16 is not written to the FIFO buffer.
;Use FIFO_Write_Circ for a circular buffer. FIFO_Write_Circ will continue
;to write bytes to a full FIFO, overwriting the oldest byte.

;Register Usage

;FIFO_Init			r16,r28,r29,r30,r31
;FIFO_Read			r16,r17,r18,r28,r29,r30,r31
;FIFO_Write			r16,r17,r18,r28,r29,r30,r31
;FIFO_Write_Circ	r16,r17,r18,r28,r29,r30,r31

;Cycle Count (assumes call and ret use for cycles each)

;					Min		Max
;FIFO_Init			25		25
;FIFO_Read			12,40*	41
;FIFO_Write			15,40*	41
;FIFO_Write_Circ	40		74?

;* Lowest minimum valid for error in read/write operations



;FIFO Macros

.macro Set_FIFO_Address		;Ram Pointer

	ldi r30,low(@0)
	ldi r31,high(@0)		;Sets FIFO address as required for
							;FIFO reads and writes

.endmacro


;FIFO Subroutines

FIFO_Init:					;Initialises a FIFO.
							;Load FIFO size into r16
							;Load FIFO Name address into Z

			std Z+5,r16		;set size

			clr r16
			st Z,r16		;clear counter

			movw YH:YL,ZH:ZL
			adiw YH:YL,6	;Y=lowest FIFO address

			std Z+1,YL
			std Z+2,YH		;set Out address

			std Z+3,YL
			std Z+4,YH		;set In address

			ret


FIFO_Read:					;Reads a byte from the FIFO starting at Z.
							;Byte is set at r16
							;Returns 0x00 if FIFO counter = 0


			ld r16,Z		;load count value

			cpi r16,0x00	;test count value for zero

			brne FIFO_Read_A

			;count = 0

			ret				;r16=0 and return


FIFO_Read_A:
			;count !=0

			dec r16			
			st Z,r16		;Count = count - 1

			ldd YL,Z+1
			ldd YH,Z+2		;load FIFO out address into Y

			ld r16,Y		;read byte from FIFO

			ldd r17,Z+5		;load FIFO size into r18:r17

FIFO_Write_Circ_MO:
			clr r18

			sub YL,r17
			sbc YH,r18		
			
			sbiw YH:YL,5	;subtract FIFO size and offset from Out_Adr

			cp YL,ZL
			cpc YH,ZH		;compare out address with Z

			breq FIFO_Read_B

			;Out Address < Highest byte in FIFO

			add YL,r17
			adc YH,r18		
			
			adiw YH:YL,6	;restore Y value and add 1

			std Z+1,YL
			std Z+2,YH		;Out address = out address + 1

			ret				;return

FIFO_Read_B:
			;Out Address = Highest byte in FIFO

			movw YH:YL,ZH:ZL;Y = Z

			adiw YH:YL,6	;Y=Y+6

			std Z+1,YL
			std Z+2,YH		;Out address = lowest out address

			ret				;return


FIFO_Write:					;Writes a byte to the FIFO starting at Z.
							;Byte is set at r16

			ld r18,Z		;load count value

			ldd YL,Z+3
			ldd YH,Z+4		;load In address to Y

			ldd r17,Z+5		;load FIFO size

			cp r17,r18		;test count value against size

			brne FIFO_Write_A

			;count = max size

			ret				;return

FIFO_Write_A:
			;count < max size

			st Y,r16		;r16 written to FIFO

			inc r18
			st Z,r18		;Count=count + 1

			clr r18

			sub YL,r17
			sbc YH,r18		
			sbiw YH:YL,5	;Y=Y-buffer size-5

			cp YL,ZL
			cpc YH,ZH		;compare in address with Z

			breq FIFO_Write_B

			;In Address < Highest FIFO byte

			add YL,r17
			adc YH,r18
			adiw YH:YL,6	;restore in address and add 1

			std Z+3,YL
			std Z+4,YH		;In Adr = In Adr + 1
			
			ret				;return

FIFO_Write_B:
			;In Address = Highest FIFO byte

			movw YH:YL,ZH:ZL
			adiw YH:YL,6	;Y = lowest FIFO byte

			std Z+3,YL
			std Z+4,YH		;In Adr = lowest FIFO byte

			ret				;return


FIFO_Write_Circ:				;Writes a byte to the FIFO starting at Z.
							;Byte is set at r16

			ld r18,Z		;load count value

			ldd YL,Z+3
			ldd YH,Z+4		;load In address to Y

			ldd r17,Z+5		;load FIFO size

			cp r17,r18		;test count value against size

			brne FIFO_Write_A

			;FIFO is full. Alter counter and Out Address

			dec r18			;lower count value to prevent errors

			push r18
			push YL
			push YH			;save counter and In address

			ldd YL,Z+1
			ldd YH,Z+2		;load FIFO out address into Y

			rcall FIFO_Write_Circ_MO

FIFO_Write_Circ_Cont:
			pop YH
			pop YL
			pop r18			;restore counter and In address

			rjmp FIFO_Write_A	;continue with new count value

